﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;
using System.Threading.Tasks;
using TMPro;
using UnityEngine;
using UnityEngine.Tilemaps;
using UnityEngine.UI;

//Utilities is a static class containing small and common utility methods
public static class Utilities
{
    /// <summary>
    /// Generates a random alphanumerical string of the desired length
    /// </summary>
    /// <param name="length">The length of the string to generate</param>
    /// <returns></returns>
    public static string GenerateRandomString(int length = 8)
    {
        string chars = "ABCDEFGHIJKLMNOPQRSTUVWXYZabcdefghijklmnopqrstuvwxyz0123456789";
        char[] stringChars = new char[length];
        System.Random random = new System.Random(Guid.NewGuid().GetHashCode());

        for (int i = 0; i < stringChars.Length; i++)
        {
            stringChars[i] = chars[random.Next(chars.Length)];
        }

        return new string(stringChars);
    }

    /// <summary>
    /// Wraps an integer into the desired range
    /// </summary>
    /// <param name="input">The integer to wrap</param>
    /// <param name="range">The range to wrap into</param>
    /// <returns>The wrapped value</returns>
    public static int Wrap(int input, int range)
    {
        return Convert.ToInt32((input % range + range) % range);
    }

    /// <summary>
    /// Generates a random city seed
    /// </summary>
    /// <returns>The generated seed</returns>
    public static string GenerateRandomCitySeed()
    {
        return GenerateRandomString();
    }

    /// <summary>
    /// Converts a city seed string into its integer representation
    /// </summary>
    /// <param name="str">The string seed to convert</param>
    /// <returns>The integer representation of the seed</returns>
    public static int GetCitySeedIntFromString(string str)
    {
        str = str.ToUpper();
        string hexStr = "";

        foreach(char c in str)
        {
            int charInt = (int)c;

            if(c >= 0x30 && c <= 0x39)
            {
                hexStr += c;
            }

            else if(c >= 0x41 && c <= 0x46)
            {
                hexStr += c;
            }

            else
            {
                int wrappedInt = Wrap(charInt, 16);

                if(wrappedInt <= 9)
                {
                    hexStr += (char)(wrappedInt + 0x30);
                }

                else
                {
                    hexStr += (char)(wrappedInt - 0xA + 0x41);
                }
            }
        }

        try
        {
            return Convert.ToInt32(hexStr, 16);
        }

        catch (Exception ex)
        {
            return -1;
        }
    }

    /// <summary>
    /// Reads new-line separated ixes from the specified file path
    /// </summary>
    /// <param name="fileResourcePath">The path to the file to read the ixes from</param>
    /// <returns>A list of ixes read from the file</returns>
    public static List<string> ReadIxes(string fileResourcePath)
    {
        TextAsset textFile = (TextAsset)Resources.Load(fileResourcePath);

        if (textFile != null)
        {
            return textFile.text.Split(new[] { "\r\n", "\r", "\n" }, System.StringSplitOptions.None).ToList();
        }

        else
        {
            Debug.LogWarning("WARNING: The text file in " + fileResourcePath + " was null - no ixes have been read. Is the path correct?");
            return new List<string>();
        }
    }

    /// <summary>
    /// Gets a random exact North, East, South or West facing rotation
    /// </summary>
    /// <returns>A float of the rotation</returns>
    public static float GetRandomNESWRotation()
    {
        Randomizer.Regenerate();
        return Randomizer.RNG.Next(0, 4) * 90;
    }

    /// <summary>
    /// Gets the position of a model in world space for a specified tile position (we have to take into account the cell gap)
    /// </summary>
    /// <param name="tilePosition">The position of the reference tile</param>
    /// <param name="modelMesh">The mesh of the model</param>
    /// <param name="rotation">The rotation of the model</param>
    /// <returns></returns>
    public static Vector3 GetModelWorldPosition(Vector3 tilePosition, Mesh modelMesh, float rotation)
    {
        rotation = Mathf.Abs(rotation);
        Vector2 roundedSize = new Vector2(0.0f, 0.0f);

        //Calculate the rounded size of the model - flip if we're rotated
        if (rotation == 90.0f || rotation == 270.0f)
        {
            roundedSize = new Vector2((float)Math.Ceiling(modelMesh.bounds.size.y), (float)Math.Ceiling(modelMesh.bounds.size.x));
        }

        else
        {
            roundedSize = new Vector2((float)Math.Ceiling(modelMesh.bounds.size.x), (float)Math.Ceiling(modelMesh.bounds.size.y));
        }

        //Now correct based on the gap
        Vector3 correctedTilePosition = new Vector3(tilePosition.x + (tilePosition.x * Constants.GridCellGap.x), tilePosition.y + (tilePosition.y * Constants.GridCellGap.y), Constants.TilemapZPosition);

        //Finally, return the position centre pivoted, based on the model size
        return new Vector3
        (
            correctedTilePosition.x + (roundedSize.x / 2.0f),
            correctedTilePosition.y + (roundedSize.y / 2.0f),
            correctedTilePosition.z
        );
    }

    /// <summary>
    /// Performs a bilinear interpolation between colours
    /// </summary>
    /// <param name="neighbours">The colours to interpolate</param>
    /// <param name="amount">The amount to interpolate</param>
    /// <returns>The interpolated colour</returns>
    public static Color BilinearInterpolate(List<Color> neighbours, float amount)
    {
        Color horColour = Color.Lerp(neighbours[0], neighbours[1], amount);
        Color verColour = Color.Lerp(neighbours[2], neighbours[3], 1.0f - amount);
        return Color.Lerp(horColour, verColour, 0.5f);
    }

    /// <summary>
    /// Computes the greatest common divisor between two integers
    /// </summary>
    /// <param name="firstVal">The first value</param>
    /// <param name="secondVal">The second value</param>
    /// <returns>The GCD value</returns>
    public static int GCD(int firstVal, int secondVal)
    {
        while (firstVal != 0 && secondVal != 0)
        {
            if (firstVal > secondVal)
            {
                firstVal %= secondVal;
            }

            else
            {
                secondVal %= firstVal;
            }
        }

        return firstVal | secondVal;
    }
}
